<?php

namespace App\Http\Controllers\Admin;

use App\Models\Admin;
use App\Http\Requests\AdminRequest;
use App\Models\Role;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class AdminController extends Controller
{

    public function index() {

        $this->authorize('view_admins', Admin::class);
        $data = Admin::orderBy('id', 'DESC')->get();

        return view('admin.admins.index', compact('data'));
    }

    public function show(Admin $admin) {

        $this->authorize('show_admins', $admin);
        return view('admin.admins.show', compact('admin'));
    }

    public function create(){

        $this->authorize('create_admins', Admin::class);

        $roles = Role::get();
        return view('admin.admins.create', compact('roles'));
    }

    public function store(AdminRequest $request) {

        $this->authorize('create_admins', Admin::class);

        $admin = Admin::create($request->all());
        $admin->roles()->sync($request->roles);
        $admin->permissions()->sync($admin->roles[0]->permissions);

        return $admin ?
            redirect()->route('admin.admins.index')->with('success', trans('messages.addOK')) :
            redirect()->route('admin.admins.index')->with('warning', trans('messages.addNO'));
    }

    public function edit(Admin $admin){

        if($admin->id == 1) {
            abort('403');
        }

        $this->authorize('edit_admins', $admin);
        $roles = Role::get();
        return view('admin.admins.edit', compact('admin', 'roles'));
    }

    public function update(AdminRequest $request, Admin $admin) {

        $this->authorize('edit_admins', $admin);
        $updated = $admin->update($request->all());
        $admin->roles()->sync($request->roles);
        $admin->permissions()->sync($admin->roles[0]->permissions);

        return $updated ?
            redirect()->route('admin.admins.index')->with('success', trans('messages.updateOK')) :
            redirect()->route('admin.admins.index')->with('warning', trans('messages.updateNO'));
    }

    public function changeStatus(Request $request){

        $updated = Admin::where('id', $request->id)->update(['status' => $request->status]);
        $url = route('admin.admins.index');

        return $updated
            ? json_encode(['code'=> '1', 'url'=> $url])
            : json_encode(['code'=> '0', 'message'=> 'نأسف لحدوث هذا الخطأ, برجاء المحاولة لاحقًا']);
    }

    public function editProfile() {

        $admin = Auth::guard('admin')->user();
        return view('admin.admins.edit_profile', compact('admin'));
    }

    public function updateProfile(Request $request) {

        $this->validate($request, [
            'name'  => 'required|max:191',
            'email' => 'required|email|max:191|unique:admins,email,'. Auth::guard('admin')->user()->id,
            'phone' => 'required|max:20|unique:admins,phone,'. Auth::guard('admin')->user()->id,
        ]);
        $updated = Auth::guard('admin')->user()->update($request->all());
        return $updated ?
            redirect()->back()->with('success', trans('messages.updateOK')) :
            redirect()->back()->with('warning', trans('messages.updateNO'));
    }

    public function changePassword() {

        $admin = Auth::guard('admin')->user();
        return view('admin.admins.change_password', compact('admin'));
    }

    public function updatePassword(Request $request) {

        $request->validate([
            'current_password'  => 'required|min:3|max:191',
            'new_password'      => 'required|min:3|max:191',
            'password_confirm'  => 'same:new_password',
        ]);

        if (!(Hash::check($request->current_password, $request->user()->password)))
            return redirect()->back()->with('danger', 'كلمة المرور التي أدخلتها لا تتطابق مع كلمة المرور الخاصة بك');

        if( strcmp($request->current_password, $request->new_password) == 0 )
            return redirect()->back()->with('danger', 'كلمة المرور الجديدة لا يمكن ان تكون مطابقة للحالية, اختر كلمة مرور اخرى');

//        update-password-finally ^^
        $updated = Auth::guard('admin')->user()->update(['password' => $request->new_password]);

        return $updated
            ? redirect()->back()->with('success', 'تم تغيير كلمة المرور بنجاح')
            : redirect()->back()->with('warning', 'حدث خطأ ما, برجاء المحاولة مرة اخرى');
    }

    public function showChangePasswordForm(Admin $admin) {

        return view('admin.admins.change_admin_password', compact('admin'));
    }

    public function updateAdminPassword(Request $request, Admin $admin) {

        $request->validate([
            'new_password'      => 'required|min:3|max:191',
            'password_confirm'  => 'same:new_password',
        ]);

        $updated = $admin->update(['password' => $request->new_password]);
        return $updated ?
            redirect()->route('admin.admins.index')->with('success', 'تم تغيير كلمة مرور مشرف بنجاح') :
            redirect()->route('admin.admins.index')->with('warning', 'حدث خطأ ما, برجاء المحاولة مرة اخرى');
    }

    public function destroy(Request $request) {

        $data = Admin::find($request->id);
        $deleted = $data->delete();
        $url = route('admin.admins.index');

        return $deleted
            ? json_encode(['code' => '1', 'url' => $url])
            : json_encode(['code' => '0', 'message' => 'نأسف لحدوث هذا الخطأ, برجاء المحاولة لاحقًا']);
    }
}
